<?php

/**
 * Mykoob Moodle integration library
 *
 * @package    mykoob_moodle
 * @author     Ilmars Poikans <ilmars.poikans@gmail.com>
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

require_once(dirname(dirname(dirname(__FILE__))) . '/config.php');
require_once($CFG->dirroot . '/lib/accesslib.php');
require_once($CFG->dirroot . '/course/lib.php');
require_once($CFG->dirroot . '/cohort/lib.php');


class MykoobMoodle {

    const ID_NUMBER_PREFIX = 'mykoob_';

    const DEBUG = false;
    const LOG_TAG = '[MYKOOB MOODLE] ';

    static $log_messages = array();

    public static function info_log($info_msg) {
        if (self::DEBUG) {
            error_log(self::LOG_TAG . $info_msg);
        }
        self::$log_messages[] = $info_msg;
    }

    public static function get_log_messages() {
        return self::$log_messages;
    }

    public static function get_log($separator = "\n") {
        return implode($separator, self::$log_messages);
    }

    public static function clear_log() {
        $log_messages = array();
    }

    public static function get_moodle_username($user_id) {
        return self::ID_NUMBER_PREFIX . $user_id;
    }

    public static function get_school_category_idnumber($school_id) {
        return self::ID_NUMBER_PREFIX . 'school_' . $school_id;
    }

    public static function get_class_cohort_idnumber($class_id) {
        return self::ID_NUMBER_PREFIX . 'class_' . $class_id;
    }


    public static function get_moodle_user($user_id) {

        global $CFG, $DB;

        $moodle_username = self::get_moodle_username($user_id);

        return $DB->get_record('user', array('username' => $moodle_username, 'mnethostid' => $CFG->mnet_localhost_id));
    }


    /**
     * Required object properties:
     *   $user->user_id
     *   $user->name
     *   $user->surname
     *   $user->email
     *   $user->city
     *   $user->country
     */
    public static function create_moodle_user($user) {

        global $CFG, $DB;

        $moodle_username = self::get_moodle_username($user->user_id);

        $mykoob_user_city = empty($user->city) ? '' : $user->city;
        $mykoob_user_country = empty($user->country_code) || $user->country_code === '---' ? 'LV' : strtoupper($user->country_code);

        $moodle_user = $DB->get_record('user', array('username' => $moodle_username, 'mnethostid' => $CFG->mnet_localhost_id));

        if (empty($moodle_user)) {

            self::info_log('Creating user ' . $moodle_username);

            $moodle_user = new stdClass();
            $moodle_user->auth = 'mykoob';
            $moodle_user->confirmed = 1;
            $moodle_user->username = $moodle_username;
            $moodle_user->password = 'not cached';
            $moodle_user->idnumber = $user->user_id;
            $moodle_user->firstname = $user->name;
            $moodle_user->lastname = $user->surname;
            $moodle_user->email = $user->email;
            $moodle_user->city = $mykoob_user_city;
            $moodle_user->country = $mykoob_user_country;
            $moodle_user->mnethostid = $CFG->mnet_localhost_id;

            $moodle_user->id = $DB->insert_record('user', $moodle_user);

            $moodle_user = $DB->get_record('user', array('username' => $moodle_username, 'mnethostid' => $CFG->mnet_localhost_id));

        } else {

            self::info_log('Updating user ' . $moodle_username);

            $moodle_user->firstname = $user->name;
            $moodle_user->lastname = $user->surname;
            $moodle_user->email = $user->email;
            $moodle_user->city = $mykoob_user_city;
            $moodle_user->country = $mykoob_user_country;

            $DB->update_record('user', $moodle_user);
        }

        return $moodle_user;
    }


    public static function get_school_category($school_id) {

        global $DB;

        $category_idnumber = self::get_school_category_idnumber($school_id);

        return $DB->get_record('course_categories', array('idnumber' => $category_idnumber));
    }

    /**
     * Required object properties:
     *   $school->school_id
     *   $school->name
     */
    public static function create_school_category($school) {

        global $DB;

        $category_idnumber = self::get_school_category_idnumber($school->school_id);

        $category = $DB->get_record('course_categories', array('idnumber' => $category_idnumber));

        if (empty($category)) {

            self::info_log('Creating course category ' . $category_idnumber . ' for school [' . $school->school_id . '] ' . $school->name);

            $category = new stdClass();
            $category->id = 0;
            $category->idnumber = $category_idnumber;
            $category->name = $school->name;
            $category->description = '';
            $category->descriptionformat = 1;
            $category->parent = 0; // top category
            $category->sortorder = 999;

            $category->id = $DB->insert_record('course_categories', $category);
            $category->context = get_context_instance(CONTEXT_COURSECAT, $category->id);

            mark_context_dirty($category->context->path);

            fix_course_sortorder();

            $category = $DB->get_record('course_categories', array('idnumber' => $category_idnumber));

        } else {
            self::info_log('Category ' . $category_idnumber . ' for school [' . $school->school_id . '] ' . $school->name . ' already exists.');
        }

        return $category;
    }


    public static function create_school_class_cohort($school, $period, $class, $context, $user_ids = array()) {

        global $DB;

        $cohort_idnumber = self::get_class_cohort_idnumber($class->class_id);

        $cohort = $DB->get_record('cohort', array('idnumber' => $cohort_idnumber));

        if (empty($cohort)) {

            self::info_log('Creating class cohort ' . $cohort_idnumber . ' for school [' . $school->school_id . '] ' . $school->name);

            $cohort = new stdClass();
            $cohort->id = 0;
            $cohort->contextid = $context->id;
            $cohort->idnumber = $cohort_idnumber;
            $cohort->name = $class->name . ' ' . $period->display_name;
            $cohort->description = $class->name . ' ' . $period->name;
            $cohort->descriptionformat = 1;
            $cohort->component = 'auth_mykoob'; // Admin can't edit, if specified
            $cohort->timecreated = time();
            $cohort->timemodified = $cohort->timecreated;

            $cohort->id = $DB->insert_record('cohort', $cohort);

            $cohort = $DB->get_record('cohort', array('idnumber' => $cohort_idnumber));

        } else {
            self::info_log('Class cohort ' . $cohort_idnumber . ' for school [' . $school->school_id . '] ' . $school->name . ' already created');
        }

        $existing_cohort_members = array_flip($DB->get_fieldset_select('cohort_members', 'userid', '', array('cohortid' => $cohort->id)));

        // self::info_log('Existing members of cohort ' . $cohort->id . ': ' . print_r($existing_cohort_members, true));

        foreach($user_ids as $user_id) {

            $moodle_user = self::get_moodle_user($user_id);

            if (isset($moodle_user)) {
                if (!array_key_exists($moodle_user->id, $existing_cohort_members)) {
                    self::info_log('Adding Mykoob user ' . $user_id . ' (' . print_r($moodle_user) . ') to cohort ' . $cohort->id);
                    cohort_add_member($cohort->id, $moodle_user->id);
                } else {
                    self::info_log('Mykoob user ' . $user_id . ' (' . print_r($moodle_user) . ') already member of cohort ' . $cohort->id);
                }
            }
        }

        return $cohort;
    }


    public function assign_roles($roles, $user_id, $context_id) {
        foreach ($roles as $role) {
            role_assign($role->id, $user_id, $context_id);
        }
    }

    public static function sync_all_schools() {

        $config = get_config('auth_mykoob');

        self::info_log('Syncing all schools...');

        if (!empty($config->schools_allowed)) {

            self::info_log('Config: Allowed schools - ' . $config->schools_allowed);

            $allowed_schools = $config->schools_allowed;
            $allowed_school_ids = empty($allowed_schools) ? array() : explode(',', $allowed_schools);

            self::info_log('Syncing schools - ' . implode(', ', $allowed_school_ids));

            foreach($allowed_school_ids as $allowed_school_id) {
                self::sync_school_data_by_school_id($allowed_school_id);
            }
        }
    }

    public static function sync_school_data_by_school_id($school_id) {

        self::info_log('Syncing school - ' . $school_id);

        if (empty($school_id)) {
            self::info_log('Can not sync empty school');
            return false;
        }

        $config = get_config('auth_mykoob');

        $mykoob_api = MykoobApi::createFromConfig($config);

        if (!$mykoob_api) {
            self::info_log('Mykoob API not configured.');
            return;
        }

        $school_periods = $mykoob_api->getSchoolPeriods($school_id)->periods;

        $last_period = end($school_periods);
        // $last_period = $school_periods[0]; // TODO: Remove

        self::info_log('Last period - ' . $last_period->period_id . ' ' . print_r($last_period, true));

        $school_data = $mykoob_api->getSchoolData($school_id, $last_period->period_id);

        self::sync_school_data($school_data->school);
    }

    public static function sync_school_data($school) {

        if (empty($school)) {
            self::info_log('Can not sync empty school');
            return false;
        }

        $period_map = array();
        $mykoob_user_map = array();
        $class_map = array();

        $moodle_user_map = array();
        $moodle_teachers = array();
        $moodle_students = array();

        $moodle_school_category = self::create_school_category($school);

        $school_category_context = context_coursecat::instance($moodle_school_category->id);

        $teacher_roles = get_archetype_roles('teacher');
        $student_roles = get_archetype_roles('student');

        $active_periods = array();

        foreach($school->periods as $period) {
            if (!$period->closed)
                $active_periods[] = $period;
            $period_map[$period->period_id] = $period;
        }

        self::info_log('Count of users: ' . count($school->users));

        foreach($school->users as $mykoob_user) {

            $moodle_user = self::create_moodle_user($mykoob_user);
            $moodle_user_map[$mykoob_user->user_id] = $moodle_user;
            $mykoob_user_map[$mykoob_user->user_id] = $mykoob_user;

            if (in_array('TEACHER', $mykoob_user->roles)) {
                self::assign_roles($teacher_roles, $moodle_user->id, $school_category_context);
                $moodle_teachers[] = $moodle_user;
            }

            if (in_array('STUDENT', $mykoob_user->roles)) {
                self::assign_roles($student_roles, $moodle_user->id, $school_category_context);
                $moodle_students[] = $moodle_user;
            }
        }

        self::info_log('Processed ' . count($school->users) . ' users');

        self::info_log('Count of periods ' . count($school->periods) . '(all) / ' . count($active_periods) . '(active)');

        self::info_log('Count of classes ' . count($school->classes));

        foreach($school->classes as $class) {
            $period = $period_map[$class->period_id];
            self::info_log('Class ' . $class->class_id . ' has period ' . $period->period_id);
            $class_cohort = self::create_school_class_cohort($school, $period, $class, $school_category_context, $class->user_ids);
            $class_map[$class->class_id] = $class;
        }
    }
}

