<?php

/**
 * Mykoob Authentication plugin
 *
 * @package    auth_mykoob
 * @author     Ilmars Poikans <ilmars.poikans@gmail.com>
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

if (!defined('MOODLE_INTERNAL')) {
    die('Direct access to this script is forbidden.'); // It must be included from a Moodle page
}

require_once($CFG->libdir . '/authlib.php');
require_once($CFG->dirroot . '/auth/mykoob/mykoob.php');
require_once($CFG->dirroot . '/auth/mykoob/mykoob_moodle.php');


class auth_plugin_mykoob extends auth_plugin_base {

    const DEBUG = false;
    const LOG_TAG = '[AUTH MYKOOB] ';

    var $log;

    /**
     * Constructor.
     */
    function auth_plugin_mykoob() {
        $this->authtype = 'mykoob';
        $this->roleauth = 'auth_mykoob';
        $this->config = get_config('auth_mykoob');
    }

    /**
     * Returns true if the username and password work and false if they are
     * wrong or don't exist. Always return true, because login is done in loginpage_hook()
     *
     * @param string $username The username (with system magic quotes)
     * @param string $password The password (with system magic quotes)
     *
     * @return bool Authentication success or failure.
     */
    function user_login($username, $password) {

        global $SESSION;

        return $username == $SESSION->mykoob_moodle_username;
    }

    /**
     * Returns true if this authentication plugin can change the users'
     * password. Mykoob plugin can't change password.
     *
     * @return bool
     */
    function can_change_password() {
        return false;
    }

    /**
     * Returns the URL for changing the users' passwords, or empty if the default
     * URL can be used.
     *
     * This method is used if can_change_password() returns true.
     * This method is called only when user is logged in, it may use global $USER.
     *
     * @return moodle_url url of the profile page or null if standard used
     */
    function change_password_url() {
        return null;
    }

    /**
     * Returns true if this authentication plugin can edit the users' profile.
     * User can edit all fields except name, last name which comes from Mykoob.
     * Check that during profile save.
     *
     * @return bool
     */
    function can_edit_profile() {
        return true;
    }

    /**
     * Returns the URL for editing the users' profile, or empty if the default URL can be used.
     * Use default profile edit URL.
     *
     * This method is used if can_edit_profile() returns true.
     * This method is called only when user is logged in, it may use global $USER.
     *
     * @return moodle_url url of the profile page or null if standard used
     */
    function edit_profile_url() {
        return null;
    }

    /**
     * Returns true if this authentication plugin is "internal".
     * Internal plugins use password hashes from Moodle user table for authentication.
     * Mykoob plugin is not internal
     *
     * @return bool
     */
    function is_internal() {
        return false;
    }

    /**
     * Indicates if password hashes should be stored in local moodle database.
     * @return bool true means md5 password hash stored in user table, false means flag 'not_cached' stored there instead
     */
    function prevent_local_passwords() {
        return true;
    }

    /**
     * Indicates if moodle should automatically update internal user
     * records with data from external sources using the information
     * from get_userinfo() method.
     *
     * @return bool true means automatically copy data from ext to user table
     */
    function is_synchronised_with_external() {
        return true;
    }

    /**
     * Updates the user's password.
     * Mykoob user can't update password. Return false.
     *
     * @param  object  $user        User table object
     * @param  string  $newpassword Plaintext password
     *
     * @return bool                 True on success
     */
    function user_update_password($user, $newpassword) {
        return false;
    }

    /**
     * Called when the user record is updated.
     * Modifies user in external database. It takes olduser (before changes) and newuser (after changes)
     * compares information saved modified information to external db.
     * Mykoob ignores update.
     *
     * @param mixed $olduser     Userobject before modifications    (without system magic quotes)
     * @param mixed $newuser     Userobject new modified userobject (without system magic quotes)
     * @return boolean true if updated or update ignored; false if error
     *
     */
    function user_update($olduser, $newuser) {
        return true;
    }

    /**
     * User delete requested - internal user record is mared as deleted already, username not present anymore.
     * Do any action in external database.
     * Mykoob ignores this
     *
     * @param object $user       Userobject before delete    (without system magic quotes)
     * @return void
     */
    function user_delete($olduser) {
        return;
    }


    /**
     * Returns true if plugin allows resetting of internal password.
     * Mykoob doesn't allow resetting of internal password.
     *
     * @return bool
     */
    function can_reset_password() {
        return false;
    }

    /**
     * Returns true if plugin allows signup.
     *
     * @return bool
     */
    function can_signup() {
        return false;
    }

    /**
     * Sign up a new user ready for confirmation.
     * Password is passed in plaintext.
     *
     * @param object $user new user object
     * @param boolean $notify print notice with link and terminate
     */
    function user_signup($user, $notify=true) {
        print_error('mykoob plugin does not support user signup', 'debug', '', 'user_signup()' );
    }

    /**
     * Returns true if plugin allows confirming of new users.
     *
     * @return bool
     */
    function can_confirm() {
        return false;
    }

    /**
     * Confirm the new user as registered.
     *
     * @param string $username
     * @param string $confirmsecret
     */
    function user_confirm($username, $confirmsecret) {
        print_error('mykoob plugin does not support user confirmation', 'debug', '', 'user_confirm()' );
    }

    /**
     * Checks if user exists in external db
     * Mykoob doesn't check that.
     *
     * @param string $username (with system magic quotes)
     * @return bool
     */
    function user_exists($username) {
        return false;
    }

    /**
     * return number of days to user password expires
     *
     * If userpassword does not expire it should return 0. If password is already expired
     * it should return negative value.
     *
     * @param mixed $username username (with system magic quotes)
     * @return integer
     */
    function password_expire($username) {
        return 0;
    }
    /**
     * Sync roles for this user - usually creator
     *
     * @param $user object user object (without system magic quotes)
     */
    function sync_roles($user) {
    }

    /**
     * Read user information from external database and returns it as array().
     * Function should return all information available. If you are saving
     * this information to moodle user-table you should honour synchronisation flags
     *
     * @param string $username username
     *
     * @return mixed array with no magic quotes or false on error
     */
    function get_userinfo($username) {
        // TODO: implement returning mykoob user_info data
        return array();
    }

    /**
     * Processes and stores configuration data for this authentication plugin.
     * @param stdClass $config
     * @return bool always true or exception
     */
    function process_config($config) {
        // This method is not called. Maybe because there is settings.php?
        return true;
    }

    /**
     * Hook for overriding behaviour of login page.
     * This method is called from login/index.php page for all enabled auth plugins.
     *
     * @global object
     * @global object
     */
    function loginpage_hook() {

//        global $frm;  // can be used to override submitted login form
//        global $user; // can be used to replace authenticate_user_login()
        global $CFG, $DB, $SESSION, $USER;

        // Prevent username from being shown on login page after logout
        // $CFG->nolastloggedin = true;

        $authprovider = optional_param('authprovider', '', PARAM_TEXT);

        if ($authprovider === 'mykoob') {

            $code = required_param('code', PARAM_TEXT);
            $state = optional_param('state', '', PARAM_TEXT);

            if (empty($code)) {
                throw new moodle_exception('error_noauthcode', 'auth_mykoob');
            }

            $mykoob_api = $this->get_mykoob_api();

            if (!$mykoob_api) {
                throw new moodle_exception('error_apinotconfigured', 'auth_mykoob');
            }

            $access_token_data = $mykoob_api->getAccessToken($code);

            if (!$access_token_data) {
                throw new moodle_exception('error_failedtogetaccesstoken');
            }

            $access_token = $access_token_data->access_token;

            if (self::DEBUG) {
                error_log(self::LOG_TAG . 'Access token data: ' . print_r($access_token_data, true));
                error_log(self::LOG_TAG . 'Access token: ' . $access_token);
            }

            $user_info_data = $mykoob_api->getUserInfo($access_token);

            if (self::DEBUG) {
                error_log(self::LOG_TAG . 'User info data: ' . print_r($user_info_data, true));
            }

            if (!$user_info_data) {
                throw new moodle_exception('error_failedtogetuserinfo');
            }

            $mykoob_user_id = $user_info_data->user_info->user_id;
            $mykoob_user_firstname = $user_info_data->user_info->name;
            $mykoob_user_lastname = $user_info_data->user_info->surname;
            $mykoob_user_email = $user_info_data->user_info->email;
            $mykoob_user_city = $user_info_data->user_info->city;
            $mykoob_user_country = $user_info_data->user_info->country_code;
            $mykoob_user_lang = 'lv'; // In the future check and filter $user_info_data->user_info->language;

            // TODO: Add sanity checks for mykoob user info fields

            if ($mykoob_user_country == '---') {
                $mykoob_user_country = 'LV';
            }

            $moodle_username = $this->config->user_prefix . $mykoob_user_id;

            $moodle_user = $DB->get_record('user', array('username' => $moodle_username, 'deleted' => 0, 'mnethostid' => $CFG->mnet_localhost_id));

            if (empty($moodle_user)) {

                // TODO: Check school ID in more optimal way

                $user_school_info_data = $mykoob_api->getUserSchoolInfo($access_token);

                if (!$user_school_info_data) {
                    throw new moodle_exception('error_failedtogetuserschoolinfo');
                }

                if (self::DEBUG) {
                    error_log(self::LOG_TAG . 'User school info data: ' . print_r($user_school_info_data, true));
                }

                $mykoob_user_school_ids = array_map(function($school) { return $school->school_id; }, $user_school_info_data->user_school_info->schools);
                $allowed_school_ids = strlen(trim($this->config->schools_allowed)) > 0 ? explode(',', $this->config->schools_allowed) : array();

                if (self::DEBUG) {
                    error_log(self::LOG_TAG . 'User school IDs: ' . print_r($mykoob_user_school_ids, true));
                    error_log(self::LOG_TAG . 'Allowed school IDs: ' . print_r($allowed_school_ids, true));
                }

                if ($this->config->allow_any_user == false && count($allowed_school_ids) > 0 && count(array_intersect($allowed_school_ids, $mykoob_user_school_ids)) === 0) {
                    throw new moodle_exception('error_notfromallowedschool');
                }

                $new_moodle_user = new stdClass();
                $new_moodle_user->auth = 'mykoob';
                $new_moodle_user->confirmed = 1;
                $new_moodle_user->username = $moodle_username;
                $new_moodle_user->password = 'not cached';
                $new_moodle_user->idnumber = $mykoob_user_id;
                $new_moodle_user->firstname = $mykoob_user_firstname;
                $new_moodle_user->lastname = $mykoob_user_lastname;
                $new_moodle_user->email = $mykoob_user_email;
                $new_moodle_user->city = $mykoob_user_city;
                $new_moodle_user->country = $mykoob_user_country;
                $new_moodle_user->lang = $mykoob_user_lang;
                $new_moodle_user->mnethostid = $CFG->mnet_localhost_id;
            }

            add_to_log(SITEID, 'auth_mykoob', '', '', $mykoob_user_id . '/' . $mykoob_user_firstname . '/' . $mykoob_user_lastname . '/' . $mykoob_user_email);

            $SESSION->mykoob_moodle_username = $moodle_username;

            $user = authenticate_user_login($moodle_username, null);

            if ($user) {

                if (!empty($new_moodle_user)) {
                    $new_moodle_user->id = $user->id;
                    $DB->update_record('user', $new_moodle_user);
                } else {
                    $moodle_user->idnumber = $mykoob_user_id;
                    $moodle_user->firstname = $mykoob_user_firstname;
                    $moodle_user->lastname = $mykoob_user_lastname;
                    $moodle_user->email = $mykoob_user_email;
                    $moodle_user->city = $mykoob_user_city;
                    $moodle_user->country = $mykoob_user_country;
                    $DB->update_record('user', $moodle_user);
                }

                complete_user_login($user);

                // Redirection
                if (user_not_fully_set_up($USER)) {
                    $urltogo = $CFG->wwwroot.'/user/edit.php';
                    // We don't delete $SESSION->wantsurl yet, so we get there later
                } else if (isset($SESSION->wantsurl) and (strpos($SESSION->wantsurl, $CFG->wwwroot) === 0)) {
                    $urltogo = $SESSION->wantsurl;    // Because it's an address in this site
                    unset($SESSION->wantsurl);
                } else {
                    // No wantsurl stored or external - go to homepage
                    $urltogo = $CFG->wwwroot.'/';
                    unset($SESSION->wantsurl);
                }
                redirect($urltogo);
            }
        }
    }

    /**
     * Post authentication hook.
     * This method is called from authenticate_user_login() for all enabled auth plugins.
     *
     * @param object $user user object, later used for $USER
     * @param string $username (with system magic quotes)
     * @param string $password plain text password (with system magic quotes)
     */
    function user_authenticated_hook(&$user, $username, $password) {
    }

    /**
     * Pre logout hook.
     * This method is called from require_logout() for all enabled auth plugins,
     *
     * @global object
     */
    function prelogout_hook() {

        global $USER;

        if ($USER->auth != 'mykoob')
            return;
    }

    /**
     * Hook for overriding behaviour of logout page.
     * This method is called from login/logout.php page for all enabled auth plugins.
     *
     * @global object
     * @global string
     */
    function logoutpage_hook() {

        global $CFG;
        global $USER;     // use $USER->auth to find the plugin used for login
        global $redirect; // can be used to override redirect after logout

        if ($USER->auth != 'mykoob')
            return;

        $redirect = $CFG->wwwroot .'/auth/mykoob/logout.php';
    }

    /**
     * Hook called before timing out of database session.
     * This is useful for SSO and MNET.
     *
     * @param object $user
     * @param string $sid session id
     * @param int $timecreated start of session
     * @param int $timemodified user last seen
     * @return bool true means do not timeout session yet
     */
    function ignore_timeout_hook($user, $sid, $timecreated, $timemodified) {
        return false;
    }

    /**
     * Returns a list of potential IdPs that this authentication plugin supports.
     * This is used to provide links on the login page.
     *
     * @param string $wantsurl the relative url fragment the user wants to get to.  You can use this to compose a returnurl, for example
     *
     * @return array like:
     *              array(
     *                  array(
     *                      'url' => 'http://someurl',
     *                      'icon' => new pix_icon(...),
     *                      'name' => get_string('somename', 'auth_yourplugin'),
     *                 ),
     *             )
     */
    function loginpage_idp_list($wantsurl) {

        $mykoob_api = $this->get_mykoob_api();

        $idp_list = array();

        if ($mykoob_api) {
            $oauth_url = $mykoob_api->getOAuthAuthorizeUrl('user_info', $this->config->callback_url);
            $name = get_string('login_idp_name', 'auth_mykoob');
            $icon = new pix_icon('i/unlock', $name);
//             $icon = new pix_icon('mykoob_icon', $name, 'auth_mykoob');
            $idp =  array('url' => new moodle_external_url($oauth_url), 'icon' => $icon, 'name' => $name);
//             $icon = new pix_icon('mykoob_logo', $name, 'auth_mykoob');
//             $icon = new pix_icon('mykoob_logo_m', $name, 'auth_mykoob');
//             $idp =  array('url' => new moodle_external_url($oauth_url), 'icon' => $icon, 'name' => '');
            $idp_list[] = $idp;
        }

        return $idp_list;
    }

    private function get_mykoob_api() {
        return MykoobApi::createFromConfig($this->config);
    }


    function cron() {

        $last_run = isset($this->config->last_cron_time) ? (int) $this->config->last_cron_time : 0;

        $current_time = time();

        if ($current_time - $last_run < 24 * 3600) { // run once a day
            return;
        }

        MykoobMoodle::info_log('Running Mykoob auth module cron job');
        MykoobMoodle::sync_all_schools();
        MykoobMoodle::info_log('Mykoob auth module cron job finished');
        $this->log = MykoobMoodle::get_log();

        $this->config->last_cron_time = $current_time;
        set_config('last_cron_time', $current_time, 'auth_mykoob');
    }

}


class moodle_external_url extends moodle_url {

    var $external_url;

    public function __construct($external_url) {
        parent::__construct($external_url);
        $this->external_url = $external_url;
    }

    public function getExternalUrl() {
        return $this->external_url;
    }

    public function out($escaped = true, array $overrideparams = null) {
        return $this->external_url;
    }
}

