<?php

/**
 * Mykoob API class
 *
 * @package    mykoob_api
 * @author     Ilmars Poikans <ilmars.poikans@gmail.com>
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

if (!function_exists('curl_init')) {
    throw new Exception('Mykoob needs the cURL PHP extension.');
}
if (!function_exists('json_decode')) {
    throw new Exception('Mykoob needs the JSON PHP extension.');
}


class MykoobApiException extends Exception
{
    protected $type;

    public function __construct($msg, $code, $type) {
        parent::__construct($msg, $code);
        $this->type = $type;
    }

    public function getType() {
        return $this->type;
    }
}


class MykoobApi {

    const API_SERVER_PROD = 'https://www.mykoob.lv';
    const API_SERVER_TEST = 'https://dev.mykoob.lv';
    const API_SERVER_DEFAULT = self::API_SERVER_PROD;

    const AUTHORIZE_ENDPOINT_PATH = '/?oauth2/authorize/';
    const TOKEN_ENDPOINT_PATH     = '/?oauth2/token/';
    const DIRECTORY_ENDPOINT_PATH = '/?oauth2/directory/';
    const RESOURCE_ENDPOINT_PATH  = '/?oauth2/resource/';

    const TIMEOUT = 10;
    const DEBUG = false;
    const CURL_DEBUG = false;
    const LOG_TAG = '[MYKOOB API] ';

    const ROLE_TEACHER       = 'TEACHER';
    const ROLE_PARENT        = 'PARENT';
    const ROLE_STUDENT       = 'STUDENT';
    const ROLE_CLASS_TEACHER = 'CLASS_TEACHER';
    const ROLE_LESSON_MASTER = 'LESSON_MASTER';
    const ROLE_SCHOOL_ADMIN  = 'SCHOOL_ADMIN';

    private $api_server;
    private $client_id;
    private $client_secret;

    private $last_exception;

    public function __construct($client_id, $client_secret, $api_server = self::API_SERVER_DEFAULT) {
        $this->api_server = rtrim($api_server, '/');
        $this->client_id = $client_id;
        $this->client_secret = $client_secret;
    }

    public function getApiServer() {
        return $this->api_server;
    }

    public function getClientId() {
        return $this->client_id;
    }

    public function getLastException() {
        return $this->last_exception;
    }

    public function getLogoutUrl($redirect_url = '') {
        return $this->api_server . '/?login/logoff/'. (empty($redirect_url) ? '' : '&redirect_uri=' . urlencode($redirect_url));
    }

    public function getOAuthAuthorizeUrl($scope, $redirect_url) {
        return $this->api_server . self::AUTHORIZE_ENDPOINT_PATH .
            '&scope=' . $scope .
            '&response_type=code' .
            '&client_id=' . urlencode($this->client_id) .
            '&redirect_uri=' . urlencode($redirect_url);
    }

    private function call($url, $args = array()) {

        $args['client_id'] = $this->client_id;
        $args['client_secret'] = $this->client_secret;

        $post_fields = http_build_query($args, '', '&'); // specify arg_separator explicitly, so changed
                                                         // arg_separator.output wouldn't cause problems

        if (self::DEBUG) {
            error_log(self::LOG_TAG . 'POST ' . $url . ' with args: ' . print_r($args, true) . "\n" .
                      self::LOG_TAG . 'POST fields: ' . print_r($post_fields, true));
        }

        $ch = curl_init($url);

        curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, self::TIMEOUT);
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, $post_fields);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_HTTPHEADER, array( 'Expect:' ) );
        curl_setopt($ch, CURLOPT_VERBOSE, self::CURL_DEBUG);

        $response = curl_exec($ch);

        if ($response === false) {
            $this->last_exception = new MykoobApiException(curl_error($ch), curl_errno($ch), 'CurlException');
            curl_close($ch);
            return false;
        }

        curl_close($ch);

        if (self::DEBUG) {
            error_log(self::LOG_TAG . 'Raw response: ' . print_r($response, true));
        }

        $data = json_decode($response);
        $json_error = json_last_error();

        if ($json_error !== JSON_ERROR_NONE) {
            $this->last_exception = new MykoobApiException('Failed to decode JSON', $json_error, 'JsonException');
            return false;
        }

        if (isset($data->error)) {
            $this->last_exception = new MykoobApiException($data->error . ': '. $data->error_description, 0, 'MykoobApiException');
            return false;
        }

        $this->last_exception = null;

        return $data;
    }

    private function call_path($endpoint_path, $args = array()) {
        return $this->call($this->api_server . $endpoint_path, $args);
    }


    public function getAccessToken($code, $args = array()) {

        $args['grant_type'] = 'authorization_code';
        $args['code'] = $code;

        return $this->call_path(self::TOKEN_ENDPOINT_PATH, $args);
    }

    public function getResource($name, $access_token, $args = array()) {

        $args['api'] = $name;
        $args['access_token'] = $access_token;

        return $this->call_path(self::RESOURCE_ENDPOINT_PATH, $args);
    }

    public function getDirectory($name, $args = array()) {

        $args['api'] = $name;

        return $this->call_path(self::DIRECTORY_ENDPOINT_PATH, $args);
    }

    public function getPersonalID($access_token) {
        return $this->getResource('personal_id', $access_token);
    }

    public function getUserInfo($access_token) {
        return $this->getResource('user_info', $access_token);
    }

    public function getUserSchoolInfo($access_token) {
        return $this->getResource('user_school_info', $access_token);
    }

    public function listAuthorizedSchools() {
        return $this->getDirectory('authorized_schools');
    }

    public function getSchoolProfile($school_id) {
        return $this->getDirectory('school_profile', array('school_id'=>$school_id));
    }

    public function getSchoolPeriods($school_id) {
        return $this->getDirectory('school_periods', array('school_id'=>$school_id));
    }

    public function getSchoolData($school_id, $period_id = null) {
        return $this->getDirectory('school_data', array('school_id'=>$school_id, 'period_id'=>$period_id));
    }

    public static function createFromConfig($config) {

        $api_server    = is_object($config) ? $config->api_server    : $config['api_server'];
        $client_id     = is_object($config) ? $config->client_id     : $config['client_id'];
        $client_secret = is_object($config) ? $config->client_secret : $config['client_secret'];

        if (!isset($api_server))
            $api_server = self::API_SERVER_DEFAULT;

        if (!isset($api_server, $client_id, $client_secret))
            return false;

        return new MykoobApi($client_id, $client_secret, $api_server);
    }
}

